## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----built--------------------------------------------------------------------
message(paste0('Datetime: ',Sys.Date(),':',Sys.time()))

## ----setup, message=FALSE-----------------------------------------------------
library(repfun)
library(dplyr)
library(kableExtra)

## ----envir--------------------------------------------------------------------
tmpdr <- tempdir()
datdir <- file.path(gsub("\\","/",tmpdr,fixed=TRUE),"datdir")
dir.create(datdir,showWarnings=FALSE)
repfun::copydata(datdir)
repfun::rs_setup(D_POP="SAFFL",D_POPLBL="Safety",
         D_POPDATA=repfun::adsl %>% dplyr::filter(SAFFL =='Y'), 
         D_SUBJID=c("STUDYID","USUBJID"), R_ADAMDATA=datdir)
repfun:::rfenv$G_POPDATA %>% dplyr::mutate(TRT01AN=ifelse(TRT01A=='Placebo',1,ifelse(TRT01A=='Xanomeline Low Dose',2,3))) %>% repfun::ru_labels(varlabels=list('TRT01AN'='Actual Treatment for Period 01 (n)')) -> G_POPDATA

## ----update-------------------------------------------------------------------
adae <- repfun:::rfenv$adamdata$adae.rda() %>% select(-SAFFL) %>% 
  repfun::ru_getdata(G_POPDATA, c("STUDYID", "USUBJID"), keeppopvars=c("TRT01AN", "TRT01A"))

## ----cntper-------------------------------------------------------------------
aesum <- repfun::ru_freq(adae,
                 dsetindenom=G_POPDATA,
                 countdistinctvars=c('STUDYID','USUBJID'),
                 groupbyvarsnumer=c('TRT01AN','TRT01A','AEBODSYS','AEDECOD'),
                 anyeventvars = c('AEBODSYS','AEDECOD'),
                 anyeventvalues = c('ANY EVENT','ANY EVENT'),
                 groupbyvarsdenom=c('TRT01AN'),
                 resultstyle="NUMERPCT",
                 totalforvar=c('TRT01AN'),
                 totalid=99,
                 totaldecode='Total',
                 codedecodevarpairs=c("TRT01AN", "TRT01A"),
                 varcodelistpairs=c(""),
                 codelistnames=list(),
                 resultpctdps=0)

## ----denorm-------------------------------------------------------------------
aesum_t <- repfun::ru_denorm(aesum,varstodenorm=c("tt_result", "PERCENT"), 
                     groupbyvars=c("tt_summarylevel", "AEBODSYS", "AEDECOD"), 
                     acrossvar="TRT01AN", acrossvarlabel="TRT01A", 
                     acrossvarprefix=c("tt_ac", "tt_p")) %>% 
  dplyr::arrange(tt_summarylevel, AEBODSYS, AEDECOD)

## ----results------------------------------------------------------------------
lbls <- sapply(aesum_t,function(x){attr(x,"label")})
knitr::kable(head(aesum_t,10), col.names=paste(names(lbls),lbls,sep=": "), 
             caption = "Denormalized Data Set for Counts and Percents") %>%  
  kable_styling(full_width = T) %>% column_spec(c(2,3), width_min = c('2in','2in'))

## ----cntper2------------------------------------------------------------------
demstats <- repfun::ru_sumstats(G_POPDATA,
                        analysisvars=c("AGE","TRTDURD"),
                        groupbyvars=c("STUDYID","TRT01AN"),
                        codedecodevarpairs=c("TRT01AN", "TRT01A"),
                        totalforvar="TRT01AN", totalid=99,
                        totaldecode="Total",
                        statsinrowsyn = "Y",
                        analysisvardps=list("AGE"=1,"TRTDURD"=2),
                        statslist=c("n", "mean", "median", "sd", "min", "max"))

## ----denorm2------------------------------------------------------------------
demprod_t <- repfun::ru_denorm(demstats, varstodenorm=c("tt_result"), 
                       groupbyvars=c("tt_avid", "tt_avnm", "tt_svid", "tt_svnm"),
                       acrossvar="TRT01AN", acrossvarlabel="TRT01A", 
                       acrossvarprefix=c("tt_ac"))

## ----results2-----------------------------------------------------------------
lbls <- sapply(demprod_t,function(x){attr(x,"label")})
knitr::kable(head(demprod_t,10), col.names=paste(names(lbls),lbls,sep=": "), 
             caption = "Denormalized Data Set for Baseline Characteristics Summary Statistics") %>%   
  kable_styling(full_width = T) %>% column_spec(c(2), width_min = c('3in'))

