# PSRICalcSM: Plant Stress Response Index Calculator - Softmax Method

<!-- badges: start -->
[![CRAN status](https://www.r-pkg.org/badges/version/PSRICalcSM)](https://CRAN.R-project.org/package=PSRICalcSM)
[![Downloads](https://cranlogs.r-pkg.org/badges/grand-total/PSRICalcSM)](https://CRAN.R-project.org/package=PSRICalcSM)
<!-- badges: end -->

`PSRICalcSM` implements the **softmax aggregation method** for calculating Plant Stress Response Index (PSRI) from time-series germination data. Built on the methodological foundation of the Osmotic Stress Response Index (OSRI) framework developed by Walne et al. (2020), it is the companion package to [`PSRICalc`](https://CRAN.R-project.org/package=PSRICalc) (geometric mean method), providing a **zero-robust** alternative that eliminates the zero-collapse failure mode.

## Why Softmax?

The geometric PSRI collapses to zero when **any** component equals zero:

```
PSRI_GM = (MSG × MRG × (1-MTG))^(1/3) × RVF
# If MRG = 0 → PSRI_GM = 0 (complete data loss)
```

The softmax PSRI handles zeros through adaptive reweighting:

```
PSRI_SM = Σ Wᵢ · Cᵢ
# If MRG = 0 → W_MRG ≈ 0, other weights renormalize
# Information from MSG and cMTG is preserved
```

In our prion-germination experiments: geometric PSRI lost 72% of barley replicates to zero-collapse; softmax PSRI retained 100%.

## Installation

```r
# From CRAN (when available)
install.packages("PSRICalcSM")

# Development version
# devtools::install_github("RFeissIV/PSRICalcSM")
```

## Quick Start

```r
library(PSRICalcSM)

# Basic PSRI_SM (3 components: MSG, MRG, cMTG)
compute_psri_sm(
  germination_counts = c(5, 15, 20),
  timepoints = c(3, 5, 7),
  total_seeds = 25
)

# With radicle vigor (4 components: MSG, MRG, cMTG, RVS)
compute_psri_sm(
  germination_counts = c(5, 15, 20),
  timepoints = c(3, 5, 7),
  total_seeds = 25,
  radicle_count = 18
)

# Detailed output with components and weights
result <- compute_psri_sm(
  germination_counts = c(5, 15, 20),
  timepoints = c(3, 5, 7),
  total_seeds = 25,
  radicle_count = 18,
  return_components = TRUE
)
result$psri_sm
result$components
result$weights
```

## Temperature Calibration

The temperature parameter `T` controls how sharply the softmax concentrates weight on dominant components. The default `T = 0.13` was calibrated via perplexity targeting (effective components ≈ 2.0 out of 3). For your own data:

```r
# Gather representative component profiles from your experiment
profiles <- list(
  control = c(MSG = 0.80, MRG = 0.90, cMTG = 0.60),
  treated = c(MSG = 0.20, MRG = 0.15, cMTG = 0.50)
)

cal <- calibrate_temperature(profiles, target_perplexity = 2.0)
cal$optimal_T

# Use calibrated T
compute_psri_sm(
  germination_counts = c(5, 15, 20),
  timepoints = c(3, 5, 7),
  total_seeds = 25,
  temperature = cal$optimal_T
)
```

## Method Selection Guide

| Criterion | Geometric (`PSRICalc`) | Softmax (`PSRICalcSM`) |
|-----------|------------------------|------------------------|
| Zero components | Collapses to 0 | Graceful degradation |
| Sample size | >25 seeds × 4 reps | Any size |
| Treatment effects | Significant expected | Any |
| Data retention | May lose replicates | 100% retention |
| Radicle integration | Discrete (1.0/1.05/1.10) | Continuous (0–1) |

## Components

| Component | Description | Range |
|-----------|-------------|-------|
| MSG | Maximum Stress-adjusted Germination | [0, 1] |
| MRG | Maximum Rate of Germination | [0, ~3] |
| cMTG | Complementary Mean Time to Germination | [0, 1] |
| RVS | Radicle Vigor Score (optional) | [0, 1] |

## Citation

If you use this package, please cite:

```
Feiss, R.A. (2026). PSRICalcSM: Plant Stress Response Index Calculator
- Softmax Method. R package version 1.0.0.
https://CRAN.R-project.org/package=PSRICalcSM
```

## Scientific Attribution

PSRICalcSM builds directly on the Osmotic Stress Response Index (OSRI) methodology established by:

**Walne, C.H., Gaudin, A., Henry, W.B., and Reddy, K.R. (2020).** In vitro seed germination response of corn hybrids to osmotic stress conditions. *Agrosystems, Geosciences & Environment*, 3(1), e20087. https://doi.org/10.1002/agg2.20087

## References

- Walne, C.H., Gaudin, A., Henry, W.B., & Reddy, K.R. (2020). In vitro seed germination response of corn hybrids to osmotic stress conditions. *Agrosystems, Geosciences & Environment*, 3(1), e20087. https://doi.org/10.1002/agg2.20087
- Feiss, R.A. (2025). PSRICalc: Plant Stress Response Index Calculator. https://CRAN.R-project.org/package=PSRICalc

## Human-AI Development Transparency

Development followed an **iterative human-machine collaboration**. All
algorithmic design, statistical methodologies, and biological validation logic
were conceptualized, tested, and iteratively refined by Richard A. Feiss
through repeated cycles of running experimental data, evaluating analytical
outputs, and selecting among candidate algorithms and approaches.

AI systems (*Anthropic Claude* and *OpenAI GPT*) served as coding assistants
and analytical sounding boards under continuous human direction, helping with:

- Code implementation and syntax validation
- Exploratory analysis of candidate approaches
- Documentation consistency and clarity
- Package compliance checking

The selection of statistical methods, evaluation of biological plausibility,
and all final methodology decisions were made by the human author. AI systems
did **not** independently originate algorithms, statistical approaches, or
scientific methodologies.

## Author

Richard A. Feiss IV, Ph.D.
Minnesota Center for Prion Research and Outreach (MNPRO)
University of Minnesota

## License

MIT © University of Minnesota
