## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE, message = FALSE,
  fig.width = 6, fig.height = 4,
  comment = "#>"
)

## ----setup, include = FALSE---------------------------------------------------
library(tipse)
library(survival)
library(dplyr)
library(ggplot2)

## ----data---------------------------------------------------------------------
knitr::kable(codebreak200[1:5, ])

## ----data-censor--------------------------------------------------------------
codebreak200 %>% count(TRT01P, CNSRRS)

## ----cox-fit------------------------------------------------------------------
cox1 <- coxph(Surv(AVAL, EVENT) ~ TRT01P, data = codebreak200)

## ----tipping-point-random, cache=TRUE-----------------------------------------
tp_random <- tipping_point_model_free(
  dat = codebreak200,
  reason = "Early dropout",
  impute = "docetaxel",
  J = 100,
  tipping_range = seq(5, 95, by = 5),
  cox_fit = cox1,
  method = "random sampling",
  seed = 12345
)

## ----tipping-point-deterministic, cache=TRUE----------------------------------
tp_deterministic <- tipping_point_model_free(
  dat = codebreak200,
  reason = "Early dropout",
  impute = "docetaxel",
  J = 100,
  tipping_range = 1:20,
  cox_fit = cox1,
  method = "deterministic sampling",
  seed = 12345
)

## ----plot-random-km-----------------------------------------------------------
plot(tp_random, type = "Kaplan-Meier")

## ----plot-deterministic-km----------------------------------------------------
plot(tp_deterministic, type = "Kaplan-Meier")

## ----plot-random-tp-----------------------------------------------------------
plot(tp_random, type = "Tipping Point")

## ----plot-deterministic-tp----------------------------------------------------
plot(tp_deterministic, type = "Tipping Point")

## ----summary-model-free-------------------------------------------------------
summary(tp_random)
summary(tp_deterministic)

## ----assess-model-bree--------------------------------------------------------
assess_plausibility(tp_random)
assess_plausibility(tp_deterministic)

## ----tipping-point-hazard-deflation, cache=TRUE-------------------------------
tp_model_based <- tipping_point_model_based(
  dat = codebreak200, reason = "Early dropout",
  impute = "docetaxel",
  imputation_model = "weibull",
  J = 100,
  tipping_range = seq(0.1, 1, by = 0.1),
  cox_fit = cox1,
  seed = 12345
)

## ----plot-model-based-KM------------------------------------------------------
plot(tp_model_based, type = "Kaplan-Meier")

## ----plot-model-based-hazard-inflation----------------------------------------
plot(tp_model_based, type = "Tipping Point")

## ----summary-model-based------------------------------------------------------
summary(tp_model_based)

## ----assess-model-based-------------------------------------------------------
assess_plausibility(tp_model_based)

## -----------------------------------------------------------------------------
knitr::kable(extenet[1:5, ])

## -----------------------------------------------------------------------------
cox2 <- coxph(Surv(AVAL, EVENT) ~ TRT01P, data = extenet)
summary_cox2 <- summary(cox2)

## Extract HR for treatment vs reference
orig_HR <- summary_cox2$coefficients[, "exp(coef)"]
orig_HR

## -----------------------------------------------------------------------------
j2r_inflation <- 1 / orig_HR
j2r_inflation

## ----cache = TRUE-------------------------------------------------------------
j2r <- tipping_point_model_based(
  dat = extenet,
  reason = "Lost to follow-up",
  impute = "neratinib", # treatment arm with dropouts
  imputation_model = "weibull",
  J = 100,
  tipping_range = j2r_inflation, # J2R through hazard inflation
  cox_fit = cox2,
  seed = 12345
)

## -----------------------------------------------------------------------------
j2r$imputation_results

